<?php
// Unitoreios Profile Service handler (UDID capture)
// Fixes: "Hồ sơ không hợp lệ" by returning a valid configuration profile (MIME application/x-apple-aspen-config)
// and properly extracting plist from PKCS#7/DER payload.

declare(strict_types=1);

// Hard-stop any accidental output (hosting includes, warnings, BOM, etc.)
error_reporting(0);
ini_set('display_errors', '0');
ini_set('zlib.output_compression', '0');
while (ob_get_level()) { @ob_end_clean(); }

header('X-Unitoreios-ProfileService: v22');

// DB is optional for UDID capture. Never break Profile Service because DB fails.
define('UNITOREIOS_SILENT_DB', true);
require_once(__DIR__ . '/layouts/config.php');


function uuidv4(): string {
    $data = random_bytes(16);
    $data[6] = chr((ord($data[6]) & 0x0f) | 0x40);
    $data[8] = chr((ord($data[8]) & 0x3f) | 0x80);
    $hex = bin2hex($data);
    return sprintf('%s-%s-%s-%s-%s',
        substr($hex, 0, 8),
        substr($hex, 8, 4),
        substr($hex, 12, 4),
        substr($hex, 16, 4),
        substr($hex, 20, 12)
    );
}


function base_url(): string {
    $host = $_SERVER['HTTP_HOST'] ?? ($_SERVER['SERVER_NAME'] ?? 'localhost');
    $host = preg_replace('/[^A-Za-z0-9\.\-:]/', '', (string)$host);
    if ($host === '') $host = 'localhost';

    $scriptName = $_SERVER['SCRIPT_NAME'] ?? '';
    $scriptDir  = str_replace('\\', '/', dirname($scriptName));
    $basePath   = rtrim($scriptDir, '/');
    if ($basePath === '/' || $basePath === '.') $basePath = '';

    // FORCE HTTPS (Profile Service + modern iOS expects HTTPS)
    return 'https://' . $host . $basePath;
}

function xml_escape(string $s): string {
    return htmlspecialchars($s, ENT_QUOTES | ENT_XML1, 'UTF-8');
}

function ident_part(string $s): string {
    // Safe ASCII identifier fragment for PayloadIdentifier.
    $s = strtolower($s);
    $s = preg_replace('/[^a-z0-9]+/', '', $s) ?? '';
    return $s !== '' ? $s : 'x';
}


function send_done_profile(string $title, string $desc, string $idreq, array $device, string $baseUrl): void {
    // IMPORTANT: iOS rejects profiles with empty PayloadContent ("Hồ sơ trống").
    // So we return a tiny WebClip payload that opens the result page (and is removable).
    $uuid = uuidv4();
    $idreqSafe = ident_part($idreq);
    $pid = 'com.unitoreios.udid.done.' . $idreqSafe . '.' . strtolower(str_replace('-', '', $uuid));

    $clipUUID = uuidv4();
    $clipId   = $pid . '.webclip';

    $title = xml_escape($title);
    $desc  = xml_escape($desc);
    $pidEsc = xml_escape($pid);
    $clipIdEsc = xml_escape($clipId);


    // Build result URL (udid_result.php?id=' . rawurlencode($idreq) . ' supports UDID/PRODUCT/VERSION/DEVICE_NAME query params)
    $qs = [];
    foreach (['UDID','PRODUCT','VERSION','DEVICE_NAME'] as $k) {
        if (!empty($device[$k]) && is_string($device[$k])) $qs[$k] = $device[$k];
    }
    $resultUrl = $baseUrl . '/udid_result.php?id=' . rawurlencode($idreq) . '';
    if (!empty($qs)) {
        $resultUrl .= '?' . http_build_query($qs, '', '&', PHP_QUERY_RFC3986);
    }

    $resultUrl = xml_escape($resultUrl);

    $label = xml_escape('Unitoreios UDID');
    $xml = <<<XML
<?xml version="1.0" encoding="UTF-8"?>
<!DOCTYPE plist PUBLIC "-//Apple//DTD PLIST 1.0//EN" "http://www.apple.com/DTDs/PropertyList-1.0.dtd">
<plist version="1.0">
<dict>
  <key>PayloadContent</key>
  <array>
    <dict>
      <key>FullScreen</key>
      <false/>
      <key>IsRemovable</key>
      <true/>
      <key>Label</key>
      <string>{$label}</string>
      <key>PayloadDescription</key>
      <string>Open Unitoreios UDID result</string>
      <key>PayloadDisplayName</key>
      <string>{$label}</string>
      <key>PayloadIdentifier</key>
      <string>{$clipIdEsc}</string>
      <key>PayloadOrganization</key>
      <string>Unitoreios</string>
      <key>PayloadType</key>
      <string>com.apple.webClip.managed</string>
      <key>PayloadUUID</key>
      <string>{$clipUUID}</string>
      <key>PayloadVersion</key>
      <integer>1</integer>
      <key>Precomposed</key>
      <true/>
      <key>URL</key>
      <string>{$resultUrl}</string>
    </dict>
  </array>

  <key>PayloadDisplayName</key>
  <string>{$title}</string>
  <key>PayloadOrganization</key>
  <string>Unitoreios</string>
  <key>PayloadIdentifier</key>
  <string>{$pidEsc}</string>
  <key>PayloadUUID</key>
  <string>{$uuid}</string>
  <key>PayloadType</key>
  <string>Configuration</string>
  <key>PayloadVersion</key>
  <integer>1</integer>
  <key>PayloadDescription</key>
  <string>{$desc}</string>
  <key>PayloadScope</key>
  <string>User</string>
</dict>
</plist>
XML;

    header_remove();
    while (ob_get_level()) { @ob_end_clean(); }
    header('X-Unitoreios-ProfileService: v22');
    ini_set('zlib.output_compression', '0');
    header('Content-Type: application/x-apple-aspen-config');
    header('Content-Disposition: attachment; filename="unitoreios_udid_done.mobileconfig"');
    header('Cache-Control: no-store, no-cache, must-revalidate, max-age=0');
    header('Pragma: no-cache');
    echo $xml;
    exit;
}

function proc_open_available(): bool {
    if (!function_exists('proc_open')) return false;
    $disabled = ini_get('disable_functions');
    if (is_string($disabled) && stripos($disabled, 'proc_open') !== false) return false;
    return true;
}

function try_php_openssl_pkcs7_verify(string $der): ?string {
    // Shared hosting fallback: decode PKCS#7/DER using PHP OpenSSL extension (no proc_open needed).
    // This extracts the embedded plist content without requiring trusted CA.
    if (!function_exists('openssl_pkcs7_verify')) return null;

    // Build a full S/MIME message so openssl_pkcs7_verify can parse it reliably.
    $b64 = chunk_split(base64_encode($der), 64, "\n");
    $smime =
        "MIME-Version: 1.0\n" .
        "Content-Type: application/pkcs7-mime; smime-type=signed-data; name=smime.p7m\n" .
        "Content-Transfer-Encoding: base64\n" .
        "Content-Disposition: attachment; filename=smime.p7m\n\n" .
        $b64;

    $in  = tempnam(sys_get_temp_dir(), 'u_p7_in_');
    $out = tempnam(sys_get_temp_dir(), 'u_p7_out_');
    $signers = tempnam(sys_get_temp_dir(), 'u_p7_sig_'); // required param (written certs)
    if (!$in || !$out || !$signers) return null;

    file_put_contents($in, $smime);

    $flags = PKCS7_NOVERIFY;
    // If available, skip signature checks entirely (we only want the content).
    if (defined('PKCS7_NOSIGS')) $flags |= PKCS7_NOSIGS;

    // Note: 3rd param must be a string path (written signers certs) in PHP 7.4.
    $ok = @openssl_pkcs7_verify($in, $flags, $signers, [], null, $out);

    $xml = null;
    if ($ok && is_file($out) && filesize($out) > 0) {
        $xml = file_get_contents($out);
    }

    @unlink($in);
    @unlink($out);
    @unlink($signers);

    if (is_string($xml) && stripos($xml, '<plist') !== false) return $xml;

    // Last resort: carve plist directly from the DER (some hosts refuse to extract content).
    $carved = carve_plist_from_blob($der);
    if (is_string($carved) && stripos($carved, '<plist') !== false) return $carved;

    return null;
}



function try_openssl_decode_der(string $der): ?string {
    if (!proc_open_available()) {
        // Try PHP OpenSSL extension fallback.
        return try_php_openssl_pkcs7_verify($der);
    }

    $tmpIn = tempnam(sys_get_temp_dir(), 'u_udid_in_');
    $tmpOut = tempnam(sys_get_temp_dir(), 'u_udid_out_');
    if (!$tmpIn || !$tmpOut) return null;
    file_put_contents($tmpIn, $der);

    $cmd = ['openssl', 'smime', '-verify', '-inform', 'der', '-in', $tmpIn, '-noverify', '-out', $tmpOut];
    $spec = [0 => ['pipe', 'r'], 1 => ['pipe', 'w'], 2 => ['pipe', 'w']];
    $p = @proc_open($cmd, $spec, $pipes, __DIR__);
    if (!is_resource($p)) {
        @unlink($tmpIn);
        @unlink($tmpOut);
        return null;
    }
    fclose($pipes[0]);

    $timeout = 8.0;
    $start = microtime(true);
    while (true) {
        $st = proc_get_status($p);
        if (!$st['running']) break;
        if ((microtime(true) - $start) > $timeout) {
            @proc_terminate($p, 9);
            break;
        }
        usleep(100000);
    }
    if (isset($pipes[1])) { @stream_get_contents($pipes[1]); @fclose($pipes[1]); }
    if (isset($pipes[2])) { @stream_get_contents($pipes[2]); @fclose($pipes[2]); }
    $code = proc_close($p);

    $xml = null;
    if ($code === 0 && is_file($tmpOut) && filesize($tmpOut) > 0) {
        $xml = file_get_contents($tmpOut);
    }
    @unlink($tmpIn);
    @unlink($tmpOut);

    if (is_string($xml) && stripos($xml, '<plist') !== false) return $xml;
    return null;
}

function carve_plist_from_blob(string $blob): ?string {
    // Some payloads may not start with '<?xml' inside the blob; search '<plist' too.
    $p1 = stripos($blob, '<?xml');
    if ($p1 === false) {
        $p1 = stripos($blob, '<plist');
    }
    $p2 = stripos($blob, '</plist>');
    if ($p1 === false || $p2 === false) return null;
    $p2 += 8;
    $xml = substr($blob, $p1, $p2 - $p1);
    return (stripos($xml, '<plist') !== false) ? $xml : null;
}

function plist_string(string $xml, string $key): ?string {
    $re = '#<key>\s*' . preg_quote($key, '#') . '\s*</key>\s*<string>\s*([^<]+)\s*</string>#i';
    if (preg_match($re, $xml, $m)) {
        return trim(html_entity_decode($m[1], ENT_QUOTES | ENT_XML1, 'UTF-8'));
    }
    return null;
}

// ---- Input validation ----
$idreq = $_GET['id'] ?? '';
$idreq = is_string($idreq) ? trim($idreq) : '';
$idreq = preg_replace('/[^a-zA-Z0-9\-_]/', '', $idreq);

if ($idreq === '') {
    // Return a valid profile anyway to avoid iOS "invalid profile".
    send_done_profile('Unitoreios - UDID', 'Thiếu ID yêu cầu. Vui lòng tạo lại UDID.', 'noid', [], base_url());
}

// iOS posts a PKCS#7 (DER) blob to this endpoint.
$raw = file_get_contents('php://input');

$logPath = __DIR__ . '/udid_debug.log';
try {
    $line = date('c') .
        ' id=' . $idreq .
        ' len=' . strlen($raw) .
        ' ct=' . ($_SERVER['CONTENT_TYPE'] ?? '') .
        ' ua=' . ($_SERVER['HTTP_USER_AGENT'] ?? '') .
        ' ip=' . ($_SERVER['REMOTE_ADDR'] ?? '') .
        "\n";
    // simple rotate to avoid bloating disk
    if (is_file($logPath) && filesize($logPath) > 1024 * 1024) {
        @rename($logPath, $logPath . '.1');
    }
    @file_put_contents($logPath, $line, FILE_APPEND);
} catch (Throwable $e) {
    // ignore
}

if (!is_string($raw) || $raw === '') {
    // Some servers may put it in a field (rare). Keep compatibility.
    $raw = isset($_POST['data']) && is_string($_POST['data']) ? $_POST['data'] : '';
}

// Try decode -> plist XML
$plistXml = null;
if (is_string($raw) && $raw !== '') {
    $plistXml = try_openssl_decode_der($raw);
    if ($plistXml === null) {
        $plistXml = carve_plist_from_blob($raw);
    }
}

$udid = $plistXml ? plist_string($plistXml, 'UDID') : null;
$deviceProduct = $plistXml ? plist_string($plistXml, 'PRODUCT') : null;
$deviceVersion = $plistXml ? plist_string($plistXml, 'VERSION') : null;
$deviceName = $plistXml ? plist_string($plistXml, 'DEVICE_NAME') : null;

// If we couldn't parse, still return a valid profile (so iOS won't show "invalid profile").
if (!$udid) {
    send_done_profile(
        'Unitoreios - UDID',
        'Không đọc được UDID (server không decode được dữ liệu). Vui lòng báo admin kiểm tra OpenSSL/proc_open.',
        $idreq,
        [],
        base_url()
    );
}

// ---- DB upsert (safe) ----
// Goal: keep a *stable* mapping between IDREQ (uuid from client) and UDID.
// Many users reinstall/reopen the app and generate a new IDREQ; if we always INSERT,
// you'll get duplicate rows with same UDID but empty email/key_id, and the API won't find keyUDID.
// This logic prefers to UPDATE an existing row for the same UDID (especially the one that already has email/key_id).
if (isset($conn) && $conn instanceof mysqli) {
    $udid = (string)$udid;
    $deviceProduct = is_string($deviceProduct) ? $deviceProduct : '';
    $deviceVersion = is_string($deviceVersion) ? $deviceVersion : '';
    $deviceName    = is_string($deviceName) ? $deviceName : '';

    $rowByIdreq = null;
    $rowByUdid  = null;

    // 1) Prefer exact match by IDREQ
    $stmt = @mysqli_prepare($conn, "SELECT id, key_id, email, devicekey, status FROM devices WHERE IDREQ = ? LIMIT 1");
    if ($stmt) {
        @mysqli_stmt_bind_param($stmt, "s", $idreq);
        @mysqli_stmt_execute($stmt);
        $rowByIdreq = function_exists('u_stmt_fetch_assoc') ? @u_stmt_fetch_assoc($stmt) : null;
        @mysqli_stmt_close($stmt);
    }

    // 2) Otherwise, reuse an existing row by UDID (prefer rows that already have email/key_id)
    $stmt2 = @mysqli_prepare(
        $conn,
        "SELECT id, key_id, email, devicekey, status
         FROM devices
         WHERE UDID = ?
         ORDER BY (email IS NOT NULL AND email <> '') DESC, (key_id IS NOT NULL) DESC, id ASC
         LIMIT 1"
    );
    if ($stmt2) {
        @mysqli_stmt_bind_param($stmt2, "s", $udid);
        @mysqli_stmt_execute($stmt2);
        $rowByUdid = function_exists('u_stmt_fetch_assoc') ? @u_stmt_fetch_assoc($stmt2) : null;
        @mysqli_stmt_close($stmt2);
    }

    $target = is_array($rowByIdreq) ? $rowByIdreq : (is_array($rowByUdid) ? $rowByUdid : null);
    $targetId = (is_array($target) && isset($target['id'])) ? intval($target['id']) : 0;

    if ($targetId > 0) {
        // Update canonical row: keep lock status; update IDREQ so API can find UDID by IDREQ.
        $stmtU = @mysqli_prepare(
            $conn,
            "UPDATE devices
             SET UDID=?, DEVICE_PRODUCT=?, DEVICE_VERSION=?, DEVICE_NAME=?, IDREQ=?
             WHERE id=?"
        );
        if ($stmtU) {
            @mysqli_stmt_bind_param($stmtU, "sssssi", $udid, $deviceProduct, $deviceVersion, $deviceName, $idreq, $targetId);
            @mysqli_stmt_execute($stmtU);
            @mysqli_stmt_close($stmtU);
        }

        // Cleanup: remove duplicate rows for same UDID that are "empty" (no email/key_id/devicekey)
        $stmtD = @mysqli_prepare(
            $conn,
            "DELETE FROM devices
             WHERE UDID = ? AND id <> ?
               AND (email IS NULL OR email = '')
               AND (key_id IS NULL OR key_id = 0)
               AND (devicekey IS NULL OR devicekey = '')"
        );
        if ($stmtD) {
            @mysqli_stmt_bind_param($stmtD, "si", $udid, $targetId);
            @mysqli_stmt_execute($stmtD);
            @mysqli_stmt_close($stmtD);
        }
    } else {
        // Insert new (first time seen UDID)
        $stmtI = @mysqli_prepare(
            $conn,
            "INSERT INTO devices (UDID, DEVICE_PRODUCT, DEVICE_VERSION, DEVICE_NAME, IDREQ, status)
             VALUES (?, ?, ?, ?, ?, 0)"
        );
        if ($stmtI) {
            @mysqli_stmt_bind_param($stmtI, "sssss", $udid, $deviceProduct, $deviceVersion, $deviceName, $idreq);
            @mysqli_stmt_execute($stmtI);
            @mysqli_stmt_close($stmtI);
        }
    }
}
// If DB is unavailable, we still return a valid profile so iOS won't fail.

send_done_profile(
    'Unitoreios - UDID OK',
    'Đã nhận UDID thành công. Bạn có thể thoát và quay lại web để kiểm tra thiết bị.',
    $idreq,
    [
        'UDID' => $udid,
        'PRODUCT' => $deviceProduct,
        'VERSION' => $deviceVersion,
        'DEVICE_NAME' => $deviceName,
    ],
    base_url()
);