<?php
// Unitoreios DB connectivity check (safe JSON)
// Endpoint: /apiunitoreios/db_test.php
// v21: fixes HTTP 500 (removed stray token), adds safe error capture + optional log.

declare(strict_types=1);

error_reporting(E_ALL);
ini_set('display_errors', '0');

$logFile = __DIR__ . '/db_test_error.log';

function dbtest_json(array $payload, int $code = 200): void {
    http_response_code($code);
    header('Content-Type: application/json; charset=utf-8');
    header('Cache-Control: no-store, no-cache, must-revalidate, max-age=0');
    header('Pragma: no-cache');
    header('X-Unitoreios-DbTest: v21');
    echo json_encode($payload, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
    exit;
}

set_error_handler(function($severity, $message, $file, $line) use ($logFile) {
    $msg = date('c') . " PHP[$severity] $message @ $file:$line\n";
    @file_put_contents($logFile, $msg, FILE_APPEND);
    return true; // prevent output
});

if (!function_exists('mysqli_connect')) {
    dbtest_json([
        'db_ok' => false,
        'error' => 'PHP thiếu extension mysqli',
        'time' => date('c'),
    ], 500);
}

try {
    // Prevent config.php from die() output
    if (!defined('UNITOREIOS_SILENT_DB')) define('UNITOREIOS_SILENT_DB', true);

    $cfg = __DIR__ . '/layouts/config.php';
    if (!is_file($cfg)) {
        dbtest_json([
            'db_ok' => false,
            'error' => 'Thiếu file layouts/config.php',
            'time' => date('c'),
        ], 500);
    }

    require $cfg;

    $ok = (isset($conn) && ($conn instanceof mysqli));

    $connectErr = function_exists('mysqli_connect_error') ? (string)mysqli_connect_error() : '';

    dbtest_json([
        'db_ok' => $ok,
        'db_host' => defined('DB_SERVER') ? DB_SERVER : null,
        'db_name' => defined('DB_NAME') ? DB_NAME : null,
        'db_user' => defined('DB_USERNAME') ? DB_USERNAME : null,
        'connect_error' => ($ok ? '' : $connectErr),
        'time' => date('c'),
        'hint' => ($ok ? 'OK' : 'Nếu connect_error rỗng, xem db_test_error.log trên hosting'),
    ], $ok ? 200 : 500);

} catch (Throwable $e) {
    @file_put_contents($logFile, date('c') . " FATAL " . $e->getMessage() . " @ " . $e->getFile() . ":" . $e->getLine() . "\n", FILE_APPEND);
    dbtest_json([
        'db_ok' => false,
        'error' => 'Lỗi PHP: ' . $e->getMessage(),
        'time' => date('c'),
    ], 500);
}
