<?php
// Cleanup old *.mobileconfig files.
// Fixes: do NOT delete the file that Safari is currently downloading.

header('Content-Type: application/json; charset=utf-8');

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    echo json_encode(['status' => 'error', 'message' => 'Invalid request method']);
    exit;
}

// "keep" may be a filename, a relative path, or a full URL (e.g. download_mobileconfig.php?f=xxx.mobileconfig)
$keep = '';
if (isset($_POST['keep']) && is_string($_POST['keep'])) {
    $raw = trim($_POST['keep']);
    if ($raw !== '') {
        $u = @parse_url($raw);
        // If it's a URL with ?f=..., prefer that.
        if (is_array($u) && !empty($u['query'])) {
            parse_str((string)$u['query'], $qs);
            if (!empty($qs['f']) && is_string($qs['f'])) {
                $keep = basename($qs['f']);
            }
        }
        // Otherwise, if the path ends with .mobileconfig, use basename(path).
        if ($keep === '' && is_array($u) && !empty($u['path']) && is_string($u['path'])) {
            $p = basename($u['path']);
            if (preg_match('/\.mobileconfig$/i', $p)) {
                $keep = $p;
            }
        }
        // Fallback: treat as raw filename/path.
        if ($keep === '') {
            $keep = basename($raw);
        }
    }
}

function cleanupMobileConfigs(string $rootDir, string $keepBase, int $olderThanSeconds): array {
    $deleted = [];
    $errors = [];

    $files = glob($rootDir . '/*.mobileconfig');
    $now = time();

    foreach ($files as $file) {
        if (!is_file($file)) continue;

        $base = basename($file);
        if ($keepBase !== '' && $base === $keepBase) {
            continue;
        }

        $mtime = @filemtime($file);
        if ($mtime === false) {
            continue;
        }

        // Only delete old files (default: > 10 minutes)
        if (($now - $mtime) < $olderThanSeconds) {
            continue;
        }

        if (@unlink($file)) {
            $deleted[] = $base;
        } else {
            $errors[] = "Error deleting file: $base";
        }
    }

    return [$deleted, $errors];
}

// Also cleanup files inside udid_profiles directory (new flow)
$deletedFiles = [];
$errors = [];

list($d1, $e1) = cleanupMobileConfigs(__DIR__, $keep, 600);
$deletedFiles = array_merge($deletedFiles, $d1);
$errors = array_merge($errors, $e1);

$keep2 = $keep; // same basename
$dir2 = __DIR__ . '/udid_profiles';
if (is_dir($dir2)) {
    list($d2, $e2) = cleanupMobileConfigs($dir2, $keep2, 600);
    $deletedFiles = array_merge($deletedFiles, $d2);
    $errors = array_merge($errors, $e2);
}

if (empty($errors)) {
    echo json_encode(['status' => 'success', 'message' => 'Cleanup completed', 'deleted_files' => $deletedFiles]);
} else {
    echo json_encode(['status' => 'error', 'message' => 'Some files could not be deleted', 'deleted_files' => $deletedFiles, 'errors' => $errors]);
}
?>
